/* --
OpenFX version 1.0 - Modelling, Animation and Rendering Package
Copyright (C) 2000 OpenFX Development Team

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

You may contact the OpenFX development team via elecronic mail
at core@openfx.org, or visit our website at http://openfx.org for
further information and support details.
-- */

/* TRISECT.CPP  */

/*-----------------9/26/2002 2:46PM-----------------
 * This algorithm is called Sqrt(3) Subdivision
 *
 *   It is different from most, in that instead of adding a
 *  new vertex along each selected edge, and splitting a face
 *  into 4, this algorithm adds a single vertex at the center
 *  of a selected (2+ vertices) face, and splits the face
 *  into 3 instead of 4.  The original selected vertices (the
 *  control mesh) is smoothed according to its original neighbors,
 *  and then each original edge ( both ends selected & not boundary )
 *  is "rotated" so that the number of edges on any vertex does
 *  not change over an iteration.
 * --------------------------------------------------*/

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <windows.h>

#ifdef __cplusplus
extern "C" {
#endif

#include "struct.h"
#include "dstruct.h"

#ifdef __cplusplus
}
#endif

#define SELECTED   1
#define FAIL      -1
#define OK         1

static HWND      hParent;
static HINSTANCE hThisInstance;

#if __WATCOMC__
int APIENTRY LibMain(HANDLE hDLL, DWORD dwReason, LPVOID lpReserved){
#else
BOOL WINAPI DllMain(HANDLE hDLL, DWORD dwReason, LPVOID lpReserved){
#endif
  switch (dwReason) {
    case DLL_PROCESS_ATTACH: {
      hThisInstance=(HINSTANCE)hDLL;
      if(hDLL == NULL)MessageBox ( GetFocus()," NULL instance",NULL, MB_OK);
      break;
    }
    case DLL_PROCESS_DETACH:
      break;
  }
  return TRUE;
}

#if __SC__
#pragma startaddress(DllMain)
#endif

typedef struct tagPOSITION{
 vector pos;
} POSITION;

static int SubdivideSelectedFaces();
static int SwapOriginalEdges( long limit_edge );
static void CalcNewPosition( vertex *vp, long vn, POSITION *temp );

#define debuglvl 2;


/*-----------------9/26/2002 11:34AM----------------
 *   MAIN DLL ENTRY POINT
 * --------------------------------------------------*/
extern "C" BOOL _Xmodeler
(HWND parent_window,HWND info_window,X__STRUCTURE *lpevi){
 lpEVI=lpevi;
 hParent=parent_window;
 {
   long lim_v, lim_e, i, nsel;
   HCURSOR hSave;
   POSITION  *SavePos=NULL, *sp;
   vertex *vp;
   char str[40];
   int mode;  // 0 = No smoothing, 1 = Smoothing

   if(NvertSelect < 3 || NvertSelect > 32000)return FALSE;
   mode = MessageBox(hParent,"Apply smoothing to selected vertices ?","Smoothing",MB_YESNOCANCEL);
   if ( mode == IDCANCEL )return FALSE;
   mode=( mode==IDYES ? 1 : 0 );
   hSave=SetCursor(LoadCursor(NULL,IDC_WAIT));
   lim_v = Nvert;
   lim_e = Nedge;

//   sprintf( str, "Smoothing Vertices (%d)...", NvertSelect );

// Allocate temporary vectors - Number of originally selected vertices
   if ( mode != 0 ){
//        MessageBox(hParent,"Allocating temporary storage space for smoothing...","DEBUGGING - MainDLL",MB_OK);
        if((SavePos=(POSITION *)X__Malloc(NvertSelect*sizeof(POSITION))) == NULL){
            MessageBox(hParent,"Unable to allocate temporary storage space for smoothing!","DEBUGGING",MB_OK);
            return FAIL;
        }
        // First Pass to Calculate new positions
        vp=MainVp;
        sp=SavePos;
        for ( i=0; i<Nvert; i++) {
            if ( vp->status == SELECTED ){
                CalcNewPosition( vp, i, sp );
                sp++;
            }
            vp++;
        }
   }

   if ( SubdivideSelectedFaces() ) {           // Subdivide each face (with 2 or more selected vertices) into 3 faces ( +1 Vert, +3 Edges, +2 Faces )
      if ( mode != 0 ){
            // Second Pass to update the Vertices with the Calculated values
//            MessageBox(hParent, str,"DEBUGGING - MainDLL",MB_OK);
            vp=MainVp;
            sp=SavePos;
            for ( i=0; i<lim_v; i++){
                if ( vp->status == SELECTED ){
                    vp->xyz[0]=sp->pos[0]; vp->xyz[1]=sp->pos[1]; vp->xyz[2]=sp->pos[2];
                    sp++;
                }
                vp++;
            }
      }
      SwapOriginalEdges( lim_e );              // "Rotate" the original edges (both end selected, & the edge is a member of 2 faces)
   }
   if(SavePos != NULL)X__Free(SavePos);
   SetCursor(hSave);
 }
 return TRUE;
}

/*-----------------9/24/2002 2:07PM-----------------
 * Subdivide the selected faces (2+ vertices selected)
 * --------------------------------------------------*/
static int SubdivideSelectedFaces(){
 face *fp;
 vertex *vp,*V0,*V1,*V2;
 long i,j, NF, count, Nv1;

 NF=Nface;
 /* Determine the number of faces that need to be split  */
 fp=MainFp;
 for ( i=0, count=0; i<Nface; i++ )
 {
     V0=(MainVp+fp->V[0]); V1=(MainVp+fp->V[1]); V2=(MainVp+fp->V[2]);
     if ((( V0->status == SELECTED ) && (V1->status == SELECTED )) ||
         (( V1->status == SELECTED ) && (V2->status == SELECTED )) ||
         (( V2->status == SELECTED ) && (V0->status == SELECTED ))) count++;
     fp++;
 }
 /* Update the Heaps to hold the extra vertices, edges, and faces.  */
 if(!UpdateVertexHeap(Nvert+count))goto EP1;
 if(!UpdateEdgeHeap(Nedge+count*3))goto EP1;
 if(!UpdateFaceHeap(Nface+count*2))goto EP1;

 /* Loop through the Faces, subdividing those with 2 or more selected vertices */
 fp=MainFp;
 for ( i=0; i<NF; i++ )
 {
     V0=(MainVp+fp->V[0]); V1=(MainVp+fp->V[1]); V2=(MainVp+fp->V[2]);  // Get the face vertex pointers
     if ((( V0->status == SELECTED ) && (V1->status == SELECTED )) ||   // Are at least 2 vertices selected?
         (( V1->status == SELECTED ) && (V2->status == SELECTED )) ||
         (( V2->status == SELECTED ) && (V0->status == SELECTED )))
     {
         CreateVertex(); Nv1=Nvert-1; vp=MainVp+Nv1;           // Create a new Vertex
         vp->xyz[0]=(V0->xyz[0]+V1->xyz[0]+V2->xyz[0])/3.0;
         vp->xyz[1]=(V0->xyz[1]+V1->xyz[1]+V2->xyz[1])/3.0;
         vp->xyz[2]=(V0->xyz[2]+V1->xyz[2]+V2->xyz[2])/3.0;
         if ( V0->gp == 1 && V1->gp == 1 && V2->gp == 1)      // If all 3 corners were glued (map attached)
         {
            vp->x = ( V0->x + V1->x + V2->x ) / 3.0;          // Set the xy mapping coordinates
            vp->y = ( V0->y + V1->y + V2->y ) / 3.0;
            vp->gp=1;
            NvertGlue++;
         }

         CreateEdge(Nv1, fp->V[0] );           // Create 3 new Edges
         CreateEdge(Nv1, fp->V[1] );
         CreateEdge(Nv1, fp->V[2] );

         CreateFace(fp->V[1],fp->V[2],Nv1);    // Preserve the same winding as the original edge
         CopyFaceProp(fp,(MainFp+Nface-1));    // Copy the face properties from the original face
         CreateFace(fp->V[2],fp->V[0],Nv1);    // Preserve the same winding as the original edge
         CopyFaceProp(fp,(MainFp+Nface-1));    // Copy the face properties from the original face
         fp->V[2]=Nv1;                         // Swap new vertex for old vertex (REUSE Face)


     };
     fp++;
 }

 return count;
 EP1:              // Unable to update the heaps!!!
 return FAIL;
}

/*-----------------9/26/2002 11:09AM----------------
 * Calculate the Alpha constant for a given valence (n)
 *
 *   This could be a array of prefilled values
 *   (* The value for valence 6 (regular) is precomputed *)
 * --------------------------------------------------*/
static double CalcAlpha ( long n ) {
 char str[40];
 double x=1.0/3.0;  // Precalc for valence = 6

    if ( n != 6 ) x= (4.0 - 2.0 * cos( 6.283185307178 / n )) / 9.0;
//    sprintf( str, "Alpha = %f, N=%d", x, n );
//    MessageBox(hParent,str,"DEBUGGING - CalcAlpha",MB_OK);
    return x;
}


/*-----------------9/26/2002 12:42PM----------------
 * Calculate the new position for the original vertices
 * --------------------------------------------------*/
static void CalcNewPosition( vertex *vp, long vn, POSITION *temp ) {

 long i, vl, vr, n;
 edge *ep;
 vertex *V0, *V1;
 double alpha;
 char str[40];

 temp->pos[0] = 0.0; temp->pos[1] = 0.0; temp->pos[2] = 0.0;  // Initialize the displacement
 ep=MainEp;
 for ( i=0,n=0; i<Nedge; i++) {                               // PLOW THROUGH all the edges looking for an edge of vp
    vl=ep->V[0]; V0=(MainVp+vl); vr=ep->V[1]; V1=(MainVp+vr);
    if ( vl == vn  || vr == vn ) {
        if ( vl == vn ) {                                     // Collect the "displacement" to the neighbors (Other end)
            temp->pos[0] += ( V1->xyz[0] - vp->xyz[0] );      //  (The mask is symmetric so use the same weight for all [1.0] )
            temp->pos[1] += ( V1->xyz[1] - vp->xyz[1] );
            temp->pos[2] += ( V1->xyz[2] - vp->xyz[2] );
        }
        else {
            temp->pos[0] += ( V0->xyz[0] - vp->xyz[0] );
            temp->pos[1] += ( V0->xyz[1] - vp->xyz[1] );
            temp->pos[2] += ( V0->xyz[2] - vp->xyz[2] );
        }
        n++;                                                  // Increment the number of neighbors ( Valence )
    }
    ep++;
 }


 if ( n > 1 ) {
    alpha = CalcAlpha( n ) / (double)n;          // Get the Alpha (weighting) constant
//    sprintf( str, "Alpha = %f", alpha );
//    MessageBox(hParent,str,"DEBUGGING - CalcNewPosition",MB_OK);

    temp->pos[0] *= alpha; temp->pos[0] += vp->xyz[0];      // Apply the displacement to vp
    temp->pos[1] *= alpha; temp->pos[1] += vp->xyz[1];
    temp->pos[2] *= alpha; temp->pos[2] += vp->xyz[2];
 }

 return;

}


/*-----------------9/26/2002 12:43PM----------------
 * Given an edge - find the 2 faces on the edge and swap the edge to the
 * other two vertices.
 * --------------------------------------------------*/
static void SwapEdge( edge *ep ) {

  long i, n, a, b, c, SaveF[2], SaveV[2];
  face *fp;

  fp=MainFp;
  a=ep->V[0];
  b=ep->V[1];                     // Get the edge vertex pointers
  for ( i=0,n=0; i<Nface; i++)  {

    if (( fp->V[0] == a ||  fp->V[1] == a ||  fp->V[2] == a ) &&    // Both of the specified vertices are on the face.
        ( fp->V[0] == b ||  fp->V[1] == b ||  fp->V[2] == b)) {
        if ( n < 2 ) {
            SaveF[n]=i;
            if (!( fp->V[0] == a ) && !( fp->V[0] == b)) c=fp->V[0];  // Find the other vertex on the face.
            if (!( fp->V[1] == a ) && !( fp->V[1] == b)) c=fp->V[1];
            if (!( fp->V[2] == a ) && !( fp->V[2] == b)) c=fp->V[2];
            SaveV[n]=c;  // SaveV[0] == x;  SaveV[1] == y;
        }
        n++;
    }
    fp++;
  }
  if ( n == 2 )
  {
     // Exchange the edge endpoints
     ep->V[0] = SaveV[0];
     ep->V[1] = SaveV[1];

     fp=MainFp+SaveF[0];                       // SaveF[0] == {a,b,x}  Replace vertex b with Other Face vertex (y)
     if ( fp->V[0] == b ) fp->V[0]=SaveV[1];
     else if ( fp->V[1] == b ) fp->V[1]=SaveV[1];
     else if ( fp->V[2] == b ) fp->V[2]=SaveV[1];

     fp=MainFp+SaveF[1];                       // SaveF[1] == {a,b,y}  Replace vertex a with Other Face vertex (x)
     if ( fp->V[0] == a ) fp->V[0]=SaveV[0];
     else if ( fp->V[1] == a ) fp->V[1]=SaveV[0];
     else if ( fp->V[2] == a ) fp->V[2]=SaveV[0];
  }
}

/*-----------------9/24/2002 2:06PM-----------------
 * Swap the original edges - both selected and edge is a member of 2 faces.
 * --------------------------------------------------*/
static int SwapOriginalEdges( long limit_e ) {

  edge *ep;
  vertex *V0, *V1;
  long i;

  ep=MainEp;
  for ( i=0; i<limit_e; i++ )      // Swap any original edges that had both ends selected
  {
     V0=(MainVp+ep->V[0]); V1=(MainVp+ep->V[1]);                     // Get the edge vertex pointers
     if (( V0->status == SELECTED ) && ( V1->status == SELECTED )) SwapEdge( ep );
     ep++;
  }


return OK;
}

