// File: DShowrenderer.cpp

// Implements the custom built renderer filter in the FilterGraph that 
// sends the output to the  OpenGL video window.

#include "video.h"
#include "DShowRenderer.h"

// functions to provide control over C style memory allocations

extern unsigned char           *ScreenL,*ScreenM,*ScreenS;
extern long                    X,Y;
extern GLfloat                 maxx,maxy;

CRenderer::CRenderer(CVideoCamera *pPres,
					 LPUNKNOWN pUnk, 
					 HRESULT *phr )
                     : CBaseVideoRenderer(__uuidof(CLSID_StereoRenderer), 
                                    NAME("Stereo Renderer"), pUnk, phr)
                                  , m_pCP( pPres){ //pointer to video object
    // Store and AddRef the texture for our use.
    ASSERT(phr);
    if (phr){
        if( !pPres ){
            *phr = E_INVALIDARG;
        }
        else {
            *phr = S_OK;
        }
    }
}

CRenderer::~CRenderer(){
    // Do nothing
}

HRESULT CRenderer::CheckMediaType(const CMediaType *pmt){
	// olny allows RGB video sample formats
    HRESULT   hr = E_FAIL;
    VIDEOINFO *pvi=0; 
    CheckPointer(pmt,E_POINTER);
    ASSERT( m_pCP);
    try    {
        // Reject the connection if this is not a video type
        if( *pmt->FormatType() != FORMAT_VideoInfo )  {
            return E_INVALIDARG;
        }
        pvi = (VIDEOINFO *)pmt->Format();
        if( IsEqualGUID( *pmt->Type(), MEDIATYPE_Video) )        {
            hr = S_OK;
            if( IsEqualGUID( *pmt->Subtype(), MEDIASUBTYPE_RGB24) ){;}
            else{
                hr = DDERR_INVALIDPIXELFORMAT;
            }
        }
        if( FAILED(hr)){
            return hr;
        }        
    }// try
    catch(...){
        Msg(TEXT("Failed to check media type in the renderer. Unhandled exception hr=0x%x"), E_UNEXPECTED);
        hr = E_UNEXPECTED;
    }   
    return hr;
}

HRESULT CRenderer::SetMediaType(const CMediaType *pmt){
    HRESULT hr = S_OK;
    VIDEOINFO *pviBmp = NULL;   // Bitmap info header
	unsigned char *S;
    try{
        // Retreive the size of this media type
        pviBmp = (VIDEOINFO *)pmt->Format();
        // Get camera frame dimensions and image mapping scale
        X = m_lVidWidth  = pviBmp->bmiHeader.biWidth;
        Y = m_lVidHeight = abs(pviBmp->bmiHeader.biHeight);
        // These factors are chosen so that the image map scales over the viewport
        // filling polygon to match.   
        maxx =(GLfloat)X/(GLfloat)BUF_SIZEX;
        maxy =(GLfloat)Y/(GLfloat)BUF_SIZEY;
        m_lVidPitch  = (m_lVidWidth * 3 + 3) & ~(3); // We are forcing RGB24
        RECT rc; GetClientRect(m_pCP->m_hwnd,&rc);
        // Allocate buffers for Video frame.
        if(ScreenL != NULL)free(ScreenL); ScreenL=NULL;
  	    if((ScreenL = (unsigned char *)malloc(3*X*Y)) == NULL){
          hr = E_UNEXPECTED;
	      return hr;
	    }
        // Allocate buffer for texture map.
        if(ScreenM != NULL)free(ScreenM); ScreenM=NULL;
  	    if((ScreenM = (unsigned char *)malloc(4*BUF_SIZEX*BUF_SIZEY)) == NULL){
          hr = E_UNEXPECTED;
	      return hr;
        }
        // Allocate buffer for image snapshot (might not be used - only for colour mode,
        // still it's easier to allocate it anyway.)
        if(ScreenS != NULL)free(ScreenS); ScreenS=NULL;
  	    if((ScreenS = (unsigned char *)malloc(3*X*Y)) == NULL){
          hr = E_UNEXPECTED;
	      return hr;
	    }
	}// try
    catch(...) {
        Msg(TEXT("Failed to set media type in the renderer. Unhandled exception hr=0x%x"), E_UNEXPECTED);
        return hr;
    }
    return hr;
}

HRESULT CRenderer::DoRenderSample( IMediaSample * pSample ){
    HRESULT hr = S_OK;
    try {   // Call through to copy the media sample for the Video camera to the RAM buffer
            hr = m_pCP->CopyMediaSample( pSample, m_lVidPitch);
    }
    catch(...)    {
    }
    return hr;
}

void Msg(TCHAR *szFormat, ...){
    TCHAR szBuffer[1024];  // Large buffer for long filenames or URLs
    const size_t NUMCHARS = sizeof(szBuffer) / sizeof(szBuffer[0]);
    const int LASTCHAR = NUMCHARS - 1;
    va_list pArgs;
    va_start(pArgs, szFormat);
    _vsntprintf(szBuffer, NUMCHARS - 1, szFormat, pArgs);
    va_end(pArgs);
    szBuffer[LASTCHAR] = TEXT('\0');
    MessageBox(NULL, szBuffer, TEXT("Stereo Video Project\0"), 
               MB_OK | MB_ICONERROR);
}
