// File: video.cpp

// Main entry point for application - and window handler functions for the
// application's main window and the OpenGL display window in which the overlaid
// output is presented.

// Camera Object method CopyMediaSample() does the Chroma-Key !!!!!

#define _WIN32_DCOM 

#include "video.h"
#include "resource.h"
#include "DShowRenderer.h"
#include "classes.h"

#pragma warning( disable : 4100 4238)


#define TIMER_ID    100    // Timer for rendering
#define TIMER_RATE   40    // milliseconds  25 fps

#define ISCHECKED(hwnd,id) (BOOL)(GetMenuState(GetMenu(hwnd), id, 0) & MF_CHECKED)
#define ALIGNULONG(i)   (((i+3)/4)*4)
#define BOUND(x,min,max) ((x) < (min) ? (min) : ((x) > (max) ? (max) : (x)))


typedef struct tagPROCESSDATA { // Structure to pass arguments to the video camera thread.
  HWND parent;
  CVideoCamera *pP;
} PROCESSDATA;

extern void SetupAviThread(void);
extern void StartAviThread(void);
extern void CheckAviStatus(void);
extern BOOL DoDialog(HINSTANCE, HWND);
extern void QuitDialog(void);
// Settings for chroma-key algorithm. (Defined in dialogs.cpp)
extern int  red_break,green_break,blue_break,snap_band;
extern BOOL red_above,green_above,blue_above,rgb_invert,use_colour;

// Global variables 
HINSTANCE              hInstance  = NULL; // Application Instance
unsigned char          *ScreenL   = NULL; // Video camera RAM buffer
unsigned char          *ScreenS   = NULL; // Snapshot buffer if using snapshot
unsigned char          *ScreenM   = NULL; // 32 bit image with MASK - mask for chroma key
long                   X=0,Y=0;           // Video frame size
unsigned char          *ScreenAVI = NULL; // Movie frame RAM buffer
long                   Xavi=0,Yavi=0;     // Movie frame dimensions
GLfloat                maxx,maxy;         // Raster scale for image mapping.
HWND                   ghWndDialog=NULL;  // Dialog box for settings.
BOOL                   bIntegrating=FALSE;// Accumulating snapshort

// Local file variables
static CVideoCamera*   g_pCameraObject = NULL; // The Camera handling Oblect
static HWND            hWndMain = NULL;        // Window properties
static long            wsizex=640,wsizey=480;
static long            timer_rate=TIMER_RATE;
static CCritSec        g_cs;                   // To prevent clash over data when rendering
static long            nIntegrations  = 25;    // Number of accumulations to average 

// Local function prototypes
static LRESULT CALLBACK AboutDlgProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
static LRESULT WINAPI MainWindowMsgProc( HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam );
static LRESULT WINAPI GLProcessWndProc(HWND, UINT, WPARAM, LPARAM );
static void GLWindowPr(void *);
static BOOL bSetupPixelFormat(HDC hDC);
static void IntegrateSnapshot(void);

INT WINAPI WinMain( HINSTANCE hInst, HINSTANCE hInstPrev, LPSTR lpCmdLine, INT nCmdShow){
    HRESULT hr = S_OK;
    hInstance = hInst;
    CoInitializeEx(NULL,COINIT_MULTITHREADED);
    // Create the camera object
    g_pCameraObject = (CVideoCamera*)malloc(sizeof(CVideoCamera));
    if( !g_pCameraObject) {
        hr = E_OUTOFMEMORY;
        Msg( TEXT("Memory allocation error: failed to create CVideoCamera object"), E_OUTOFMEMORY);
        return 0L;
    }
    // Create the AVI player object
    SetupAviThread(); 
    g_pCameraObject->m_hwnd=NULL;    
    // Declare window classes.
    WNDCLASSEX wc = { sizeof(WNDCLASSEX), CS_CLASSDC, MainWindowMsgProc, 0L, 0L,
                      GetModuleHandle(NULL), 
                      LoadIcon(hInstance, MAKEINTRESOURCE(IDI_TEXTURES)), 
                      NULL,(HBRUSH)GetStockObject(WHITE_BRUSH), NULL,
                      CLASSNAME, NULL };
    RegisterClassEx( &wc );
    WNDCLASSEX wc1 = { sizeof(WNDCLASSEX), CS_CLASSDC, GLProcessWndProc, 0L, 0L,
                      GetModuleHandle(NULL), 
                      LoadIcon(hInstance, MAKEINTRESOURCE(IDI_TEXTURES)), 
                      NULL, NULL, NULL,
                      CLASSNAME1, NULL };
    if(!RegisterClassEx(&wc1))MessageBox(NULL,"Class 1 NOT Registered",NULL,MB_OK);
    // Create the application window
    hWndMain = CreateWindow( CLASSNAME, CLASSNAME,
                           WS_OVERLAPPEDWINDOW, 0, 0, 200, 100,
                           GetDesktopWindow(), NULL, wc.hInstance, NULL );
    HMENU hMenuMain = GetSystemMenu(hWndMain, FALSE);
    BOOL rc = AppendMenu(hMenuMain, MF_SEPARATOR, 0, NULL);
    rc = AppendMenu(hMenuMain, MF_STRING | MF_ENABLED, 
                    ID_HELP_ABOUT, 
                    TEXT("About Chroma ...\0"));
    rc = AppendMenu(hMenuMain, MF_STRING | MF_ENABLED, 
                    ID_CONFIG_VIDEO, 
                    TEXT("Configure the Capture Device...\0"));
    ShowWindow( hWndMain, SW_SHOWDEFAULT );
    DoDialog(hInstance,hWndMain); // Make the settings Dialog.
    UpdateWindow( hWndMain );
    // Make the working threads.
    PROCESSDATA p1;
    p1.parent=hWndMain; 
    p1.pP=g_pCameraObject; 
    // Execute the Camera live video thread. This thread is responsible for
    // rendering the output using OpenGL and performing the chroma-key task.
    _beginthread(GLWindowPr,0,(void *)(&p1));
    // Exectute the thread that plays the movie.
    StartAviThread();
    // The main message loop - the two spawned threads also have their
    // own message loops.
    MSG msg;
    ZeroMemory( &msg, sizeof(msg) );
    while(1){
     if(PeekMessage(&msg,NULL,0,0,PM_NOREMOVE)){
        GetMessage(&msg,NULL,0,0);
        if(msg.message == WM_QUIT)break;
        TranslateMessage(&msg);
        DispatchMessage(&msg);
     }
     else{
       Sleep(0);
     }
    }
    // Close down gracefully - freeing everything.
    if(g_pCameraObject && IsWindow(g_pCameraObject->m_hwnd))
      SendMessage(g_pCameraObject->m_hwnd, WM_CLOSE, 0, 0);
    QuitDialog(); // Close the Modeless Dialog box.
    Sleep(1000);  // for W2000 bug ???
    if(ScreenL != NULL)free(ScreenL);
    if(ScreenM != NULL)free(ScreenM);
    if(ScreenS != NULL)free(ScreenS);
    UnregisterClass( CLASSNAME, hInstance );
    UnregisterClass( CLASSNAME1, hInstance );
    CoUninitialize();
    if( g_pCameraObject ){ free(g_pCameraObject);  g_pCameraObject = NULL;  }
    return 0L;
}

static void GLWindowPr(void *arg){
 // Yhis function is the main thread for the camera video acquisition
 HRESULT hr = S_OK;
 HWND hwnd,hWndParent;
 UINT uTimerID=0;
 PROCESSDATA *pp = (PROCESSDATA *)arg;
 hWndParent = (HWND)pp->parent;
 hwnd = CreateWindow( // Make the OpenGL output window.
     CLASSNAME1,CLASSNAME1,
     WS_POPUP | WS_CAPTION | WS_SYSMENU | WS_THICKFRAME,
     300, 200, 
     wsizex+GetSystemMetrics(SM_CXBORDER)+GetSystemMetrics(SM_CYSIZE),
     wsizey+GetSystemMetrics(SM_CYBORDER)*2,
     hWndParent,NULL,hInstance,NULL);
 pp->pP->m_hwnd=hwnd;
 SetWindowLongPtr(hwnd, GWLP_USERDATA, (LONG_PTR)pp);
 pp->pP->initializeGL(hwnd);
 // Build the FilterGraph to render the camera video into the RAM buffer.
 hr = pp->pP->InitDShowRenderer();
 if( FAILED(hr)){ 
   MessageBox(NULL,"Failed to initialize",NULL,MB_OK); 
   DestroyWindow(hwnd);
   _endthread();
   return; 
 }
 ShowWindow(hwnd,SW_SHOW);
 UpdateWindow(hwnd);
 // Start the timer that will be used to initiate the rendering process.
 uTimerID = (UINT) SetTimer(hwnd, TIMER_ID,  timer_rate, NULL);
 if(uTimerID == 0)MessageBox(NULL,"Failed to create timer",NULL,MB_OK);
 // message processing loop
 MSG msg;
 while(1){
     if(PeekMessage(&msg,NULL,0,0,PM_NOREMOVE)){
        GetMessage(&msg,NULL,0,0);
        TranslateMessage(&msg);
        DispatchMessage(&msg);
     }
     else{
       Sleep(0);
     }
 }
 // Clean up and exit this thread.
 KillTimer(hwnd, TIMER_ID);
 pp->pP->Cleanup();
 pp->pP->m_hwnd=NULL;
 _endthread();
 return;
}

static LRESULT WINAPI MainWindowMsgProc( HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam ){
    switch( msg )    { // Process messages for the Application Window
       case WM_PAINT:{
              HDC hDC;
              PAINTSTRUCT  ps;
              hDC = BeginPaint(hWnd, &ps);
              char strx[]="Chroma Key Application";
              TextOut(hDC,0,0,strx,strlen(strx));
              EndPaint(hWnd, &ps);
            }
            break;
		case WM_DESTROY:
			PostQuitMessage( 0 );
            return 0;
        case WM_CHAR:        {
            if (wParam == VK_ESCAPE){
              PostMessage(hWnd, WM_CLOSE, 0, 0);
            }
        }
        break;
        case WM_SYSCOMMAND: {
            switch (wParam){
                case ID_HELP_ABOUT:
                    CreateDialog(hInstance, MAKEINTRESOURCE(IDD_ABOUTBOX), hWnd, 
                                (DLGPROC) AboutDlgProc);
                    return 0;
                case ID_CONFIG_VIDEO:  // Configure the Video Camera
                    if(g_pCameraObject)g_pCameraObject->ConfigSourceFilter(hWnd);
                    return 0;
            }
        }
        break;
    }
    return DefWindowProc( hWnd, msg, wParam, lParam );
}

static LRESULT WINAPI GLProcessWndProc( HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam ){
    PROCESSDATA *p;
    switch( msg )    {    // Handle messages for the Output Window
        case WM_CREATE:{  
            HGLRC hRC;
            HDC hDC;
            hDC = GetDC(hWnd);
            bSetupPixelFormat(hDC);
            hRC = wglCreateContext( hDC );
            wglMakeCurrent( hDC, hRC );
//            initializeGL(hWnd);
          }
          break;
		case WM_DESTROY:{
              HGLRC  hRC;
              HDC    hDC;
              hRC = wglGetCurrentContext();
              hDC = wglGetCurrentDC();
              wglMakeCurrent(NULL, NULL);
              if (hRC != NULL)wglDeleteContext(hRC);
              if (hDC != NULL)ReleaseDC(hWnd, hDC);
  			  PostQuitMessage( 0 );
              return 0;
			}
        case WM_TIMER:  // Render the window using an OpenGL textured quad, this is
            if(bIntegrating)IntegrateSnapshot();
        case WM_PAINT:  // handled withing the Video Camera Rendering Object 
            p=(PROCESSDATA *)GetWindowLongPtr( hWnd, GWLP_USERDATA);
            if(p){
              if(p->pP){
                p->pP->Render(hWnd);
              }
            }
            break;
         case WM_SIZE:
           // If the window size changes we need to change the pixel aspect ratio
           // on the movie (the 3D geometry on the textured mesh will change as we
           // change the Viewport.) 
           if(Xavi > 0 && Yavi > 0)glPixelZoom((GLfloat)LOWORD(lParam)/(GLfloat)Xavi,
                                   (GLfloat)HIWORD(lParam)/(GLfloat)Yavi);
           glViewport(0, 0, LOWORD(lParam), HIWORD(lParam));
           break;
         case WM_SYSCOMMAND:
            switch(LOWORD(wParam & 0xfff0)){
              case SC_CLOSE:
                PostQuitMessage(0);   // only for case when THREADED
                break;
              default:
                break;
            }
            break;
         case (WM_USER+1):{  // If we change to a different movie we will also have to adjust
             RECT rc;        // to pixel scaling to accommodate different size. This USER message
             GetClientRect(hWnd,&rc);  // is send from the movie renderer filter.
             if(Xavi > 0 && Yavi > 0)glPixelZoom((GLfloat)rc.right/(GLfloat)Xavi,
                                                 (GLfloat)rc.bottom/(GLfloat)Yavi);
           } 
           break;
         default:
            break;
    }
    return DefWindowProc( hWnd, msg, wParam, lParam );
}

static void IntegrateSnapshot(void){
 // Calculate the running average over several frames.  Xa(n) = ( X(n)+(n-1)/n* Xa(n-1))/n
 static float count= 2.0;
 if(ScreenL != NULL && ScreenS != NULL && X > 0 && Y > 0){
   long i;
   unsigned char *s,*l;
   for(i=0,s=ScreenS,l=ScreenL;i<X*Y*3;i++,s++,l++){
     *s = (unsigned char)(((float)((float)(*l)+(count-1.0)*(((float)(*s)))))/count);
   }
 }
 count++;
 if(count > (float)nIntegrations){
   count = 1.0; bIntegrating=FALSE;
   MessageBeep(MB_OK);
  }
}

///////////////////  Video Camera Object /////////////////////

CVideoCamera::CVideoCamera()
: m_hwnd( NULL )
{
}

CVideoCamera::~CVideoCamera(){}

void CVideoCamera::Cleanup(void){
    // Shut down the video capture graph
    if( m_pMC ){        m_pMC->Stop();    }
}

// Build and run the FilterGraph to Render the Video from the Camera
HRESULT CVideoCamera::InitDShowRenderer(void){
    HRESULT hr = S_OK;
    CComPtr<IBaseFilter>    pRenderer;      // Renderer Filter
    CRenderer               *pCTR=0;        // DShow renderer object
   try {
        // Create the filter graph
        hr = m_pGB.CoCreateInstance(CLSID_FilterGraph, NULL, CLSCTX_INPROC);
        if( FAILED(hr)){
            Msg(TEXT("Failed to create filter graph. hr = 0x%08x"), hr);
            return hr;
        }
        // Get the graph's media control and media event interfaces
        m_pGB.QueryInterface(&m_pMC);
        m_pGB.QueryInterface(&m_pME);
		// Get the graph's streams Interface
		hr = m_pGB.QueryInterface(&m_pGS);
		if(FAILED(hr)){
            Msg(TEXT("Failed to get interface to GraphStreams. hr = 0x%08x"), hr);
		}
        // Create the custom Renderer object to render into the RAM buffer
        pCTR = new CRenderer(this, NULL, &hr);
        if (FAILED(hr) || !pCTR)  {
            Msg(TEXT("Could not create texture renderer object.  hr=0x%x"),hr);
            return hr;
        }
        pRenderer = pCTR;
		// Get a pointer to the IBaseFilter interface on the TextureRenderer
        // and add it to the existing graph
        hr = m_pGB->AddFilter(pRenderer, L"Stereo Renderer");
        if (FAILED(hr))   {
            Msg(TEXT("Could not add renderer filter to graph.  hr=0x%x"), hr);
            return hr;
        }
        // Initialize the video capture devices to render the incoming video    
        hr = CaptureVideo( pRenderer);
        if (FAILED(hr)){
            // CaptureVideo will display an appropriate error message
            return hr;
        }
		// Try to sync the video sources - this code does not make any
        // difference so it is commented out.
		// Start the graph running
        hr = m_pMC->Run();
        if (FAILED(hr))  {
            Msg(TEXT("Could not run the DirectShow graph.  hr=0x%x"), hr);
            return hr;
        }
    }// try
    catch(...) {
        Msg(TEXT("Application encountered an unexpected error.\r\n\r\n"));
        hr = E_UNEXPECTED;  
    }
    return hr;
}

HRESULT CVideoCamera::CaptureVideo(IBaseFilter *pRenderer){ // Render the video
    HRESULT hr = S_OK;
	long    nfilters=0;
    try {
        // Create the capture graph builder object to assist in building
        // the video capture filter graph
        hr = CoCreateInstance (CLSID_CaptureGraphBuilder2 , NULL, CLSCTX_INPROC,
                               IID_ICaptureGraphBuilder2, (void **) &(m_pCapture.p));
        if (FAILED(hr)){
            Msg(TEXT("Could not create the capture graph builder!  hr=0x%x\0"), hr);
            return hr;
        }
        // Attach the existing filter graph to the capture graph
        hr = m_pCapture->SetFiltergraph(m_pGB);
        if (FAILED(hr))        {
            Msg(TEXT("Failed to set capture filter graph!  hr=0x%x\0"), hr);
            return hr;
        }
        // Use the system device enumerator and class enumerator to find
        // a video capture/preview device, such as a desktop USB video camera.
        hr = FindCaptureDevice(&m_pSrcFilter,&nfilters);
		//Msg(TEXT("%ld filters found"),nfilters);
		if (FAILED(hr))        {
            // Don't display a message because FindCaptureDevice will handle it
            return hr;
        }
           // Add the returned capture filter to our graph.
        hr = m_pGB->AddFilter(m_pSrcFilter, L"Video Capture");
        if (FAILED(hr))     {
            Msg(TEXT("Could not add the capture filter to the graph.  hr=0x%x\r\n\r\n"));
            return hr;
        }
		// Render the preview pin on the video capture filter.
        // This will create and connect any necessary transform filters.
        // We pass a pointer to the IBaseFilter interface of our CRenderer
        // video renderer, which will draw the incoming video onto an OpenGL window.
		// Render PREVIEW pin does not allow syncronism.
        hr = m_pCapture->RenderStream (&PIN_CATEGORY_CAPTURE, &MEDIATYPE_Video,
                                       m_pSrcFilter, NULL, pRenderer);
        if (FAILED(hr))        {
            Msg(TEXT("Could not render the capture stream.  hr=0x%x\r\n\r\n"));
            return hr;
        }
	}// try
    catch(...)    {
        Msg(TEXT("Application encountered an unexpected error when trying to render the graph."));
        hr = E_UNEXPECTED;
    }
    return S_OK;
}

HRESULT CVideoCamera::FindCaptureDevice(IBaseFilter ** ppSrcFilter, // Identify the video
											   long *nfilters){     // source we want to usr.
    HRESULT hr = S_OK;
    CComPtr <IMoniker> pMoniker;
    CComPtr <ICreateDevEnum> pDevEnum;
    CComPtr <IEnumMoniker> pEnum;
    IBaseFilter *pSrc = NULL;
	IMoniker *pMoniker1 = NULL;
    ULONG cFetched;
    if (!ppSrcFilter) return E_POINTER;
    try    {
        // Create the system device enumerator
        hr = CoCreateInstance (CLSID_SystemDeviceEnum, NULL, CLSCTX_INPROC,
                            IID_ICreateDevEnum, (void **) &pDevEnum);
        if (FAILED(hr)) {
            Msg(TEXT("Couldn't create system device enumerator.  hr=0x%x"), hr);
            return hr;
        }
        hr = pDevEnum->CreateClassEnumerator (CLSID_VideoInputDeviceCategory, &pEnum, 0);
        if (FAILED(hr)) {
            Msg(TEXT("Couldn't create class enumerator for video input device category.  hr=0x%x"), hr);
            return hr;
        }
        // If there are no enumerators for the requested type, then 
        // CreateClassEnumerator will succeed, but pClassEnum will be NULL.
        if (pEnum == NULL) {
            Msg(TEXT("No video capture device was detected.\r\n\r\n"));
            return E_FAIL;
        }
		while (pEnum->Next(1, &pMoniker, NULL) == S_OK){
          IPropertyBag *pPropBag;
          hr = pMoniker->BindToStorage(0, 0, IID_IPropertyBag, 
                                       (void**)(&pPropBag));
          if (FAILED(hr)){
            continue;  // Skip this one, maybe the next one will work.
          } 
          // Find the description or friendly name.
          VARIANT varName;
          VariantInit(&varName);
          hr = pPropBag->Read(L"Description", &varName, 0);
          if (FAILED(hr)){
             hr = pPropBag->Read(L"FriendlyName", &varName, 0);
          }
          if (SUCCEEDED(hr))    {
            USES_CONVERSION;
            char ttt[128]; sprintf(ttt,"Use this device ?");
    	    if(IDYES == MessageBox(NULL,OLE2T(varName.bstrVal),ttt,MB_YESNO)){
		  	  if(pMoniker1 == NULL)pMoniker1=pMoniker;
		    }
            VariantClear(&varName); 
          }
          pPropBag->Release();
        }
		if(pMoniker1 != NULL){
          *nfilters=1;
		  hr = pMoniker1->BindToObject(0,0,IID_IBaseFilter, (void**)&pSrc);
          if (FAILED(hr)){
              Msg(TEXT("Couldn't bind moniker to filter object.  hr=0x%x"), hr);
              return hr;
          }
        }
        *ppSrcFilter  = pSrc; 
    }// try
    catch(...)    {
        Msg(TEXT("Application encountered an unexpected error when trying to find capture device."));
        hr = E_UNEXPECTED;
    }
    return hr;
}

void CVideoCamera::CheckMovieStatus(void){ // Check the Video filter for events.
    long lEventCode;                       // We don't really want to do anything here.
    long lParam1;
    long lParam2;
    HRESULT hr = S_OK;
    if (!m_pME){ return; }    
    // Check for completion events
    hr = m_pME->GetEvent(&lEventCode, (LONG_PTR *) &lParam1, (LONG_PTR *) &lParam2, 0);
    if (SUCCEEDED(hr)){
        // Free any memory associated with this event
        hr = m_pME->FreeEventParams(lEventCode, lParam1, lParam2);
    }
}

// Configure the Video Camera - useful for setting up Chroma-Key behaviour.
// All standard DirectShow stuff.
void CVideoCamera::ConfigSourceFilter(HWND hWnd){
IBaseFilter *pFilter=m_pSrcFilter;
if(!pFilter)return;
/* Obtain the filter's IBaseFilter interface. (Not shown) */
ISpecifyPropertyPages *pProp;
HRESULT hr = pFilter->QueryInterface(IID_ISpecifyPropertyPages, (void **)&pProp);
if (SUCCEEDED(hr)){
    // Get the filter's name and IUnknown pointer.
    FILTER_INFO FilterInfo;
    hr = pFilter->QueryFilterInfo(&FilterInfo); 
    IUnknown *pFilterUnk;
    pFilter->QueryInterface(IID_IUnknown, (void **)&pFilterUnk);

    // Show the page. 
    CAUUID caGUID;
    pProp->GetPages(&caGUID);
    pProp->Release();
    OleCreatePropertyFrame(
        hWnd,                   // Parent window
        0, 0,                   // Reserved
        FilterInfo.achName,     // Caption for the dialog box
        1,                      // Number of objects (just the filter)
        &pFilterUnk,            // Array of object pointers. 
        caGUID.cElems,          // Number of property pages
        caGUID.pElems,          // Array of property page CLSIDs
        0,                      // Locale identifier
        0, NULL                 // Reserved
    );
    // Clean up.
    pFilterUnk->Release();
    FilterInfo.pGraph->Release(); 
    CoTaskMemFree(caGUID.pElems);
    hr = m_pMC->Stop();
    Sleep(1000);
    hr = m_pMC->Run();
}
return;
}

HRESULT CVideoCamera::Render(HWND hWnd){ // Render the Video Supeimposed on the AVI 
    HRESULT hr = S_OK;
    CAutoLock lock(&g_cs);
    try    {
       HDC hDC;
       HDC hDC1;
       PAINTSTRUCT  ps;
       hDC = BeginPaint(hWnd, &ps);
       glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
       // First Render the AVI frame buffer 
       if(ScreenAVI != NULL)glDrawPixels((GLsizei)Xavi,(GLsizei)Yavi,GL_RGB,
                              GL_UNSIGNED_BYTE,(GLvoid *)ScreenAVI);
       // Now Draw the Camera Output by rendering a masked (semi transparent)image map.
       // The transparency (alpha channel) setting is dictated by the Chroma Key algorithm.
       if(ScreenM != NULL){ // ScreenM points at the Video RAM buffer with 4th byte used for transparency.
         makeMap(ScreenM,BUF_SIZEX,BUF_SIZEY);
         DrawGeometryWithMask();
       }
       glFlush();
       glFinish();
       hDC1 = wglGetCurrentDC();
       SwapBuffers(hDC1);
       EndPaint(hWnd, &ps); 
		// Check to see if any of the graphs have events pending. (Not really necessary.)
		CheckMovieStatus();
        if(ScreenAVI != NULL)CheckAviStatus();
        hr = S_OK;
	}// try
    catch(...){
        Msg(TEXT("Application encountered an unexpected error when trying to render the scene."));
        hr = E_UNEXPECTED;
    }
    return hr;
}

void CVideoCamera::makeMap(unsigned char *pixels, int x, int y){
 // Make the image map from the array of pixels byte 4 is the ALPHA channel (transparency)
 glBindTexture(GL_TEXTURE_2D,2);
 glTexImage2D(GL_TEXTURE_2D,0,4,x,y,0,GL_RGBA,GL_UNSIGNED_BYTE,(GLvoid *)pixels);
 glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
 glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
 glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
 glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
}

void CVideoCamera::DrawGeometryWithMask(void){
 // Render a single texured QUAD polygon that fills the viewport. Scale the texture coordinates
 // so that the image fills the visible viewport, this is required because of OpenGL's 
 // requirement to have power of 2 texture sizes, as discussed in the book. 
 // Scale Factors "maxx" and "maxy" stretch the power of two texture so that the video frame image
 // is stretched out to fill the viewport.
 glBindTexture(GL_TEXTURE_2D,2);
 glEnable(GL_TEXTURE_2D); 
 glEnable(GL_BLEND);
 glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
 glBegin(GL_QUADS);
 glTexCoord2f(0.0,0.0);
 glVertex3f(0.0,0.0,0.0);
 glTexCoord2f(maxx,0.0);
 glVertex3f(1.0,0.0,0.0);
 glTexCoord2f(maxx,maxy);
 glVertex3f(1.0,1.0,0.0);
 glTexCoord2f(0.0,maxy);
 glVertex3f(0.0,1.0,0.0);
 glEnd();
 glDisable(GL_BLEND);
 glDisable(GL_TEXTURE_2D);
}


// THIS FUNCTION IMPLEMENTS THE CHROMA KEY OPERATION BY MIXING THE VIDEO
// BUFFER WITH THE AVI BUFFER WHICH IS THEN RENDERED IN RESPONSE TO THE TIMER
// AND THE WM_PAINT MESSAGE
//
// This is called from the camera filtergraph's renderer filter.
HRESULT CVideoCamera::CopyMediaSample( IMediaSample *pSample, 
		                 LONG lSamplePitch){
    HRESULT hr = S_OK;
    CAutoLock lock(&g_cs);
    BYTE * pSampleBuffer = NULL;
    unsigned char *S = NULL,*M = NULL,*T=NULL;
    unsigned char R,G,B,Rs,Gs,Bs;
	UINT row;
    UINT col;
    LONG  lTexturePitch;     // Pitch of texture
	if( !pSample ) return E_POINTER;
    try {
      // Get the video bitmap buffer
      hr = pSample->GetPointer( &pSampleBuffer );
      if( FAILED(hr)){return hr;}
      S=ScreenL; M=ScreenM; T=ScreenS;
      // Copy the video sample into the RAM buffer pointed to by "ScreenL"
      // and to the texture map "ScreenM" where the ALPHA byte is used to make 
      // the map transparent so that the pixels in the movie are seen.
	  if(use_colour && X > 0 && Y > 0 && S != NULL){
        for(row = 0; row < Y; row++ ) {
          BYTE *pBmpBufferOld = pSampleBuffer;
          for (col = 0; col < X; col++)  {
            R = pSampleBuffer[2];
            G = pSampleBuffer[1];
            B = pSampleBuffer[0];
            *S++ = R; *S++ = G; *S++ = B;
            *M++ = R; *M++ = G; *M++ = B;
            // If RGB satisfies constraints then show camera else show movie.
            if(((red_above && R >= red_break)     || (!red_above && R <= red_break)) &&
               ((green_above && G >= green_break) || (!green_above && G <= green_break)) &&
               ((blue_above && B >= blue_break)   || (!blue_above && B <= blue_break))){
              if(rgb_invert)*M=0;
              else          *M=255; 
            }  
            else{
              if(rgb_invert)*M=255;
              else          *M = 0; 
            }
            M++;
            pSampleBuffer += 3;
          }
          M += (BUF_SIZEX-X)*4;
          pSampleBuffer  = pBmpBufferOld + lSamplePitch;
        }
      }  
	  else if(T != NULL && X > 0 && Y > 0 && S != NULL){ // Use snapshot
        for(row = 0; row < Y; row++ ) {
          BYTE *pBmpBufferOld = pSampleBuffer;
          for (col = 0; col < X; col++)  {
            R = pSampleBuffer[2];
            G = pSampleBuffer[1];
            B = pSampleBuffer[0];
            *S++ = R; *S++ = G; *S++ = B;
            *M++ = R; *M++ = G; *M++ = B;
            Rs = *T++; Gs = *T++; Bs = *T++;
            // Factor in the snapshot
            if( abs((int)R - (int)Rs) < snap_band 
             && abs((int)G - (int)Gs) < snap_band 
             && abs((int)B - (int)Bs) < snap_band)
                 *M=0;     // show movie
            else *M=255;   // show video
            M++;
            pSampleBuffer += 3;
          }
          M += (BUF_SIZEX-X)*4;
          pSampleBuffer  = pBmpBufferOld + lSamplePitch;
        }
      }
    }
    catch(...) {;}
    return hr;
}

GLvoid CVideoCamera::initializeGL(HWND hWnd){
 glClearColor(0.0,0.0,0.0,1.0);
 glClearDepth(1.0);
 glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
 glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
 glMatrixMode(GL_PROJECTION);
 glLoadIdentity();
 // Ortho view in unit square.
 glOrtho(0.0,1.0,0.0,1.0,-1.0,1.0);
 glMatrixMode( GL_MODELVIEW );
 glRasterPos2i(0,0);
 glPixelZoom((GLfloat)1.0,(GLfloat)1.0);
 makeCheckeredImageMap();  // Dummy map.
 return;
}
#define checkImageWidth 64
#define checkImageHeight 64

GLubyte checkImage[checkImageWidth][checkImageHeight][4];

void CVideoCamera::makeCheckeredImageMap(void){
 GLfloat parameter3[]={GL_SPHERE_MAP};
 int i, j, c, d, e;
 for (i = 0; i < checkImageWidth; i++) {
   for (j = 0; j < checkImageHeight; j++) {
     c = ((((i&0x8)==0)^((j&0x8))==0))*255;
     checkImage[i][j][0] = (GLubyte) c;
     checkImage[i][j][1] = (GLubyte) c;
     checkImage[i][j][2] = (GLubyte) c;
     checkImage[i][j][3] = (GLubyte) c/2;
   }
 }
 glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
 glPixelStorei(GL_PACK_ALIGNMENT,1);
 glBindTexture(GL_TEXTURE_2D,1);
 glTexImage2D(GL_TEXTURE_2D, 0, 4, checkImageWidth,
              checkImageHeight, 0, GL_RGBA, GL_UNSIGNED_BYTE,
              &checkImage[0][0][0]);
 glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
 glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
 glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
 glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
 glHint(GL_PERSPECTIVE_CORRECTION_HINT,GL_NICEST);
 glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
 glTexGenfv(GL_S,GL_TEXTURE_GEN_MODE,parameter3);
 glTexGenfv(GL_T,GL_TEXTURE_GEN_MODE,parameter3);
}

/////////////////////////// End of Camera Object //////////////////

static BOOL bSetupPixelFormat(HDC hDC){
  static PIXELFORMATDESCRIPTOR pfd = {
    sizeof(PIXELFORMATDESCRIPTOR),    // size of this pfd
      1,                              // version number
      PFD_DRAW_TO_WINDOW |            // support window
      PFD_SUPPORT_OPENGL |            // support OpenGL
      PFD_DOUBLEBUFFER,               // double buffered
      PFD_TYPE_RGBA,                  // RGBA type
      24,                             // 24-bit color depth
      0, 0, 0, 0, 0, 0,               // color bits ignored
      0,                              // no alpha buffer
      0,                              // shift bit ignored
      0,                              // no accumulation buffer
      0, 0, 0, 0,                     // accum bits ignored
      32,                             // 32-bit z-buffer
      0,                              // no stencil buffer
      0,                              // no auxiliary buffer
      PFD_MAIN_PLANE,                 // main layer
      0,                              // reserved
      0, 0, 0                         // layer masks ignored
      };
  int pixelformat;
  if ( (pixelformat = ChoosePixelFormat(hDC, &pfd)) == 0 ) {
    MessageBox(NULL, "ChoosePixelFormat failed", "Error", MB_OK);
    return FALSE;
  }
  if (SetPixelFormat(hDC, pixelformat, &pfd) == FALSE) {
    MessageBox(NULL, "SetPixelFormat failed", "Error", MB_OK);
    return FALSE;
  }
  return TRUE;
}

static LRESULT CALLBACK AboutDlgProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam){
    switch (message){
        case WM_INITDIALOG:
            return TRUE;
        case WM_COMMAND:
            if (wParam == IDOK){
                EndDialog(hWnd, TRUE);
                return TRUE;
            }
            break;
    }
    return FALSE;
}
