// File: aviplay.cpp

// The functions in this file implement a thread that plays the AVI movie
// which appears in the background of the ChromaKey. 

// The AVI movie is played using a C++ object of class "CAviPlay"

#define _WIN32_DCOM 

#include "aviplay.h"
#include "avirenderer.h"
#include "classes.h"

#pragma warning( disable : 4100 4238)

// Useful macros
#define ALIGNULONG(i)   (((i+3)/4)*4)
#define BOUND(x,min,max) ((x) < (min) ? (min) : ((x) > (max) ? (max) : (x)))
#define X__Malloc(s)    LocalAlloc(LMEM_FIXED,s)
#define X__Free(p)      LocalFree((HLOCAL)p)

extern void Msg(TCHAR *szFormat, ...);

extern HINSTANCE               hInstance;   // application instance
extern unsigned char           *ScreenAVI;  // RAM buffer for AVI
extern long                    Xavi,Yavi;   // Resolutions of AVI movie


typedef struct tagPROCESSDATAAVI { // Structure to pass several parameters to THREAD
  CAviPlay *pP;
} PROCESSDATAAVI;

static void AviThread(void *arg);
static void ReleaseAviThread(void);

static CAviPlay*  g_pAviObject = NULL;   //pointer to our AVI movie player object
static char MediaFile[255];

#include "fileopen.cpp"

///////////////  Functions to create AVI player object ///////////////

void SetupAviThread(void){ // Create the AVI object.
 g_pAviObject = (CAviPlay*)malloc(sizeof(CAviPlay));
}


void StartAviThread(void){
  if(!GetFileName(MediaFile,hInstance,NULL))return;
  static PROCESSDATAAVI p1; // so doesn't get deleted when function returns
  p1.pP=g_pAviObject; 
  _beginthread(AviThread,0,(void *)(&p1));
}

void CheckAviStatus(void){ // Pass call to object's method.
 if(g_pAviObject)g_pAviObject->CheckMovieStatus();
}

///////////////////////////////////////////////////////////////////////////////

static void AviThread(void *arg){ 
 // this function is the main thread for the AVI player - it 
 HRESULT hr = S_OK;
 PROCESSDATAAVI *pp = (PROCESSDATAAVI *)arg;
 hr = pp->pP->InitAviPlay();
 if( FAILED(hr)){ 
   MessageBox(NULL,"Failed to initialize AVI thread",NULL,MB_OK); 
   _endthread();
   return; 
 }
 MSG msg;
 while(1){
     if(PeekMessage(&msg,NULL,0,0,PM_NOREMOVE)){
        GetMessage(&msg,NULL,0,0);
        TranslateMessage(&msg);
        DispatchMessage(&msg);
     }
     else{
       Sleep(0);
     }
 }
 pp->pP->Cleanup();
 ReleaseAviThread();
 _endthread();
 return;
}

static void ReleaseAviThread(void){
 if( g_pAviObject ){ 
   free(g_pAviObject);  
   g_pAviObject = NULL;  
 }
 if(ScreenAVI != NULL)X__Free(ScreenAVI); ScreenAVI=NULL;
}

///////////////////  AVI player object's methods  //////////////////

CAviPlay::CAviPlay()
{
}

CAviPlay::~CAviPlay()
{
    Cleanup();
}

void CAviPlay::Cleanup(void){
    // Shut down the graph
    if( m_pMC ){
        m_pMC->Stop();
    }
}

HRESULT CAviPlay::InitAviPlay(){            // Create the FilterGraph
    HRESULT hr = S_OK;
    CComPtr<IBaseFilter>    pRenderer;      // Texture Renderer Filter
    CComPtr<IBaseFilter>    pRenderer2;     // Texture Renderer Filter 2
    CComPtr<IBaseFilter>    pFSrc;          // Source Filter
    CComPtr<IPin>           pFSrcPinOut;    // Source Filter Output Pin   
    CAviRenderer            *pCTR=0;        // DShow Texture renderer
   try {
        hr = m_pGB.CoCreateInstance(CLSID_FilterGraph, NULL, CLSCTX_INPROC);
        if( FAILED(hr)){
            Msg(TEXT("Failed to create filter graph. hr = 0x%08x"), hr);
            return hr;
        }
        // Get the graph's media control, media event, media seeking and media Filter 
        // interfaces
        m_pGB.QueryInterface(&m_pMC);
        m_pGB.QueryInterface(&m_pME);
        m_pGB.QueryInterface(&m_pMS);  // media Seeking 
        m_pGB.QueryInterface(&m_pMP);  // media Position 
        m_pGB.QueryInterface(&m_pMF);  // can be used to prevent frame loss
        // Create the Avi Renderer object
        pCTR = new CAviRenderer(this, NULL, &hr);
        if (FAILED(hr) || !pCTR)  {
            Msg(TEXT("Could not create texture renderer object 1.  hr=0x%x"), hr);
            return hr;
        }
        pRenderer = pCTR;
		// Get a pointer to the IBaseFilter interface on the Renderer
        // and add it to the existing graph
        hr = m_pGB->AddFilter(pRenderer, L"Avi Renderer");
        if (FAILED(hr))   {
            Msg(TEXT("Could not add renderer filter to graph.  hr=0x%x"), hr);
            return hr;
        }
        // Build the movie playing FilterGraph
        hr = BuildMovieGraph(pRenderer);
        if (FAILED(hr))return hr;
        // Start the graph running
        hr = m_pMC->Run();
        if (FAILED(hr))  {
            Msg(TEXT("Could not run the DirectShow graph.  hr=0x%x"), hr);
            return hr;
        }
    }// try
    catch(...) {
        Msg(TEXT("Application encountered an unexpected error.\r\n\r\n"));
        hr = E_UNEXPECTED;  
    }
    return hr;
}

HRESULT CAviPlay::BuildMovieGraph(IBaseFilter *pRenderer){
	USES_CONVERSION;
    HRESULT hr = S_OK;
	long    nfilters=0;
    CComPtr<IBaseFilter> pSrcFilter;
    CComPtr<IBaseFilter> pInfTee;
    try {
        // Create the capture graph builder object to assist in building
        // the video capture filter graph
        hr = CoCreateInstance (CLSID_CaptureGraphBuilder2 , NULL, CLSCTX_INPROC,
                               IID_ICaptureGraphBuilder2, (void **) &(m_pCapture.p));
        if (FAILED(hr)){
            Msg(TEXT("Could not create the capture graph builder!  hr=0x%x\0"), hr);
            return hr;
        }
        // Attach the existing filter graph to the capture graph
        hr = m_pCapture->SetFiltergraph(m_pGB);
        if (FAILED(hr))        {
            Msg(TEXT("Failed to set capture filter graph!  hr=0x%x\0"), hr);
            return hr;
        }
        hr=m_pGB->AddSourceFilter(A2W(MediaFile),L"Source File",&pSrcFilter);
        if (FAILED(hr))    {
          Msg(TEXT("Failed to set Input Filename!%s,  hr=0x%x"), MediaFile,hr);
          return hr;
        }
        hr = CoCreateInstance (CLSID_InfTee, NULL, CLSCTX_INPROC,
                         IID_IBaseFilter, (void **) &pInfTee);
        hr=m_pGB->AddFilter(pInfTee,L"Inf TEE");
        if (FAILED(hr))    {
           Msg(TEXT("Failed to add the InfTEE filter: hr=0x%x"), hr);
          return hr;
        }
        hr = m_pCapture->RenderStream(0,0,pSrcFilter,0,pInfTee);
        if (FAILED(hr)){Msg("Fail to connect input pin"); return hr;}
        IPin *pPin2;
        hr = m_pCapture->FindPin(pInfTee,PINDIR_OUTPUT,0,0,TRUE,0,&pPin2); 
        if (FAILED(hr)){Msg("Fail find pin2"); return hr;}

        hr = m_pCapture->RenderStream (0, &MEDIATYPE_Video,
                                       pPin2, NULL, pRenderer);
        if (FAILED(hr))        {
            Msg(TEXT("Could not render the AVIcapture stream.  hr=0x%x\r\n\r\n"));
            return hr;
        }
	}// try
    catch(...)    {
        Msg(TEXT("Application encountered an unexpected error when trying to render the graph."));
        hr = E_UNEXPECTED;
    }
    return S_OK;
}

// This method is called from the movie RENDERER filter
HRESULT CAviPlay::CopyMediaSample( IMediaSample *pSample, LONG lSamplePitch){
    HRESULT hr = S_OK;
    BYTE * pSampleBuffer = NULL;
    unsigned char *S = NULL;
	UINT row;
    UINT col;
    LONG  lTexturePitch;     // Pitch of texture
	if( !pSample ) return E_POINTER;
    try {
        // Get the video bitmap buffer
        hr = pSample->GetPointer( &pSampleBuffer );
        if( FAILED(hr)){return hr;}
		S=ScreenAVI;         // Buffer to receive the AVI video frame
		if(Xavi > 0 && Yavi > 0 && S != NULL){
          for(row = 0; row < Yavi; row++ ) {
            BYTE *pBmpBufferOld = pSampleBuffer;
            for (col = 0; col < Xavi; col++)  {
              *S++ = pSampleBuffer[2];
              *S++ = pSampleBuffer[1];
              *S++ = pSampleBuffer[0];
    		  pSampleBuffer += 3;
            }
            pSampleBuffer  = pBmpBufferOld + lSamplePitch;
		  }
        }

    }
    catch(...) {;}
    return hr;
}

void CAviPlay::CheckMovieStatus(void){ // Check for movie coming to the end and restart it.
    long lEventCode;
    long lParam1;
    long lParam2;
    HRESULT hr = S_OK;
    if (!m_pME){ return; }    
    // Check for completion events
    while(SUCCEEDED(m_pME->GetEvent(&lEventCode, (LONG_PTR *) &lParam1, 
                   (LONG_PTR *) &lParam2, 0)))    {
        switch (lEventCode){
          case EC_COMPLETE:
            if(!m_pMP)Msg(TEXT("Bad Positioning pointer"));
            else{
              hr = m_pMP->put_CurrentPosition(0);
              if (FAILED(hr))  {
                Msg(TEXT("Could not Set position.  hr=0x%x"), hr);
              }
            }
            hr = m_pMC->Run();
            if (FAILED(hr))  {
              Msg(TEXT("Could not rstart the DirectShow graph.  hr=0x%x"), hr);
            }
            break;
        }
        hr = m_pME->FreeEventParams(lEventCode, lParam1, lParam2);
    }
}
